<?php
use App\Models\Member;
use App\Models\Membership;
use Carbon\Carbon;

/**
 * Format a date string into the specified format.
 *
 * @param  string  $string  The date string to be formatted.
 * @return string           The formatted date string.
 */
if (!function_exists('format_date')) {
	function format_date($string = '2023-06-19 19:05:02')
	{
		$date = Carbon::parse($string);
		$formatted_date = $date->format(get_option('date_format', 'd-m-Y'));
															
		return $formatted_date;
	}
}

/**
 * Format the current date into the specified format.
 *
 * @param  string  $string  The date string to be formatted.
 * @return string           The formatted date string.
 */

if (!function_exists('format_current_date')) {
    function format_current_date($format)
    {
        return Carbon::now()->format($format);
    }
}

/**
 * Get the expiry date by adding a specified number of days to the given date.
 *
 * @param  string  $date  The start date.
 * @param  int     $days  The number of days to add.
 * @return string         The formatted expiry date string.
 */
if (!function_exists('get_expiry_date')) {
	function get_expiry_date($date = '2023-06-19 19:05:02', $days = 0)
	{
		$start_date = Carbon::parse($date);

		$expiry_date = $start_date->addDays($days);
		$formatted_date = $expiry_date->format('d F Y');
		return $formatted_date;
	}
}

/**
 * Format an invoice number by padding with zeros to the left.
 *
 * @param  string  $number  The invoice number to be formatted.
 * @return string           The formatted invoice number.
 */
if (!function_exists('format_invoice_no')) {
	function format_invoice_no($number = '0')
	{
		return str_pad($number, 6, '0', STR_PAD_LEFT);
	}
}

/**
 * Get the value of a setting from the settings collection.
 *
 * @param  \Illuminate\Database\Eloquent\Collection  $settings  The collection of settings.
 * @param  string                                     $name      The name of the setting.
 * @param  mixed                                      $optional  The optional default value.
 * @return mixed                                                The value of the setting or the optional default value.
 */
if (!function_exists('get_setting')) {
	function get_setting($settings, $name, $optional = '')
	{
		$row = $settings->firstWhere('name', $name);
		if ($row != null) {
			return $row->value;
		}
		return $optional;
	}
}

/**
 * Get the value of an option from the cache or database settings table.
 *
 * @param  string  $name      The name of the option.
 * @param  mixed   $optional  The optional default value.
 * @return mixed             The value of the option or the optional default value.
 */
if (!function_exists('get_option')) {
	function get_option($name, $optional = '')
	{ 
		$value = Cache::get($name);
		$setting = DB::table('settings')->where('name', $name)->get();
		if (!$setting->isEmpty()) {
			$value = $setting[0]->value;
		} else {
			$value = $optional;
		}
		return $value;
	}
}

/**
 * Get the currency symbol for a given currency code.
 *
 * @param  string  $currency_code  The currency code.
 * @return string                  The currency symbol.
 */
if (!function_exists('get_currency_symbol')) {
	function get_currency_symbol($currency_code)
	{
		include app_path() . '/Helpers/currency_symbol.php';

		if (array_key_exists($currency_code, $currency_symbols)) {
			return $currency_symbols[$currency_code];
		}
		return $currency_code;
	}
}

/**
 * Get the currency list for a given currency code.
 *
 * @return array                  The currency list.
 */
if (!function_exists('get_currency_list')) {
	function get_currency_list()
	{
		include app_path() . '/Helpers/currency_list.php';
		return $currency_list;
	}
}

/**
 * Get the list of date formats
 *											
 * @return array 				The date formats.
 */
if (!function_exists('get_date_formats')) {
	function get_date_formats()
	{
		include app_path() . '/Helpers/date_formats.php';
		return $date_formats;
	}
}

/**
 * Get the list of timezones
 *											
 * @return array 				The date formats.
 */
if (!function_exists('get_timezones')) {
	function get_timezones()
	{
		include app_path() . '/Helpers/timezones.php';
		return $timezones;
	}
}

/**
 * Get the currency symbol based on the currency option.
 *
 * @return string  The currency symbol.
 */
if (!function_exists('currency_symbol')) {
	function currency_symbol()
	{
		$currency_code = get_option('currency', 'USD');
		$currency_symbol = get_currency_symbol($currency_code);
		$currency_symbol = html_entity_decode($currency_symbol);

		return $currency_symbol;
	}
}

/**
 * Format an amount with currency symbol.
 *
 * @param  string  $amount  The amount to be formatted.
 * @return string          The formatted amount with currency symbol.
 */
if (!function_exists('format_amount')) {
	function format_amount($amount = '0')
	{
		$currency_code = get_option('currency', 'USD');
		$currency_symbol = get_currency_symbol($currency_code);
		$currency_symbol = html_entity_decode($currency_symbol);
		
		return $currency_symbol . ' ' . number_format(floatval($amount), 2);
	}
}

/**
 * Round off an amount to two decimal places.
 *
 * @param  mixed  $amount  The amount to be rounded off.
 * @return string          The rounded off amount.
 */
if (!function_exists('round_off')) {
	function round_off($amount = 0)
	{
		return number_format(floatval($amount), 2);
	}
}

/**
 * Get the list of permissions.
 *
 * @return array  The list of permissions.
 */
if (!function_exists('get_permissions')) {
	function get_permissions()
	{
		include app_path() . '/Helpers/permissions.php';
		return $permissions_list;
	}
}

/**
 * Get the profile picture URL for a user.
 *
 * @param  mixed   $user      The user or user ID.
 * @param  string  $optional  The optional default image.
 * @return string             The URL of the profile picture.
 */
if (!function_exists('get_profile_picture')) {
	function get_profile_picture($user, $optional = 'profile-placeholder.jpg')
	{
		$user = \App\Models\User::find($user);
		$image_name = $user->profile_picture;
		if (!empty($image_name)) {
			return asset('images/profile/' . $image_name);
		} else {
			return asset('images/' . $optional);
		}
	}
}

/**
 * Get the URL for an image, with an optional default image.
 *
 * @param  string  $image_name  The name of the image.
 * @param  string  $optional    The optional default image.
 * @return string               The URL of the image.
 */
if (!function_exists('img_url')) {
	function img_url($image_name, $optional = 'profile-placeholder.jpg')
	{
		if (!empty($image_name)) {
			return asset('images/profile/' . $image_name);
		} else {
			return asset('images/' . $optional);
		}
	}
}

/**
 * Get the URL for the logo, with a default logo if not set.
 *
 * @return string  The URL of the logo.
 */
if (!function_exists('logo_url')) {
	function logo_url($option)
	{
		$logo = get_option($option);
		if (!empty($logo)) {
			return asset('images/' . $logo);
		} else {
			return asset('images/default_logo.png');
		}
	}
}

/**
 * Get the name of the user role.
 *
 * @param  mixed   $role  The user role.
 * @return string         The name of the user role.
 */
if (!function_exists('user_role')) {
	function user_role($role = "")
	{
		if (!empty($role)) {
			$role = $role->name;
		}

		return $role;
	}
}

/**
 * Get the expired memberships.
 *
 * @return \Illuminate\Support\Collection  The collection of expired memberships.
 */
if (!function_exists('get_expired_memberships')) {
	function get_expired_memberships()
	{
		return DB::table('members')
			->join('memberships', 'members.id', '=', 'memberships.member_id')
			->where('memberships.created_at', function ($query) {
				$query->select('created_at')
					->from('memberships')
					->whereColumn('member_id', 'members.id')
					->orderByDesc('created_at')
					->limit(1);
			})
			->where('memberships.valid_to', '<', now())
			->where('members.status', 'active')
			->orderByDesc('memberships.valid_to')
			->get();
	}
 
}

/**
 * Get the unpaid members.
 *
 * @return \Illuminate\Support\Collection  The collection of unpaid members.
 */
if (!function_exists('get_unpaid_members')) {
	function get_unpaid_members()
	{
		return DB::table('members')
			->where('members.status', 'active')
			->join('memberships', 'members.id', '=', 'memberships.member_id')
			->join('membership_types', 'memberships.membership_type_id', '=', 'membership_types.id')
			->leftJoin('transactions', 'memberships.id', '=', 'transactions.transactionable_id')
			->select(
				'members.id',
				'members.name',
				'members.profile_picture',
				'memberships.id AS membership_id',
				DB::raw('COALESCE(SUM(CASE WHEN transactions.dr_cr = "dr" THEN transactions.amount ELSE 0 END), 0) AS actual_amount'),
				DB::raw('COALESCE(SUM(CASE WHEN transactions.dr_cr = "cr" THEN transactions.amount ELSE 0 END), 0) AS total_paid'),
				DB::raw('COALESCE(SUM(CASE WHEN transactions.dr_cr = "dr" THEN transactions.amount ELSE 0 END), 0) - COALESCE(SUM(CASE WHEN transactions.dr_cr = "cr" THEN transactions.amount ELSE 0 END), 0) AS due_amount')
			)
			->groupBy('members.id', 'members.name', 'members.profile_picture', 'membership_id')
			->havingRaw('total_paid < actual_amount')->orderBy('memberships.created_at', 'DESC')->get();
	}
}

if (!function_exists('replace_short_code')) {
	function replace_short_code($body, $replace_data = [])
	{
		$body = str_replace('<br>', '', $body);

		foreach ($replace_data as $key => $value) {
			$body = str_replace('{{' . $key . '}}', $value, $body);
		}
		return $body;
	}
}

/**
 * Get the country codes from the JSON file.
 *
 * @return array  The array of country codes.
 */
if (!function_exists('get_country_codes')) {
	function get_country_codes()
	{
		return json_decode(file_get_contents(app_path() . '/Helpers/country.json'), true);
	}
}

/**
 * Get the dial code for a given country.
 *
 * @param  string  $country  The name of the country.
 * @return string|null       The dial code for the country, or null if not found.
 */
if (!function_exists('get_dial_code')) {
	function get_dial_code($country)
	{
		$country_codes = json_decode(file_get_contents(app_path() . '/Helpers/country.json'), true);

		foreach ($country_codes as $item) {
			if ($item['country'] === $country) {
				return $item['dial_code'];
			}
		}
	}
}

/**
 * Pluralize a duration.
 *
 * @param  int  $duration  The duration value.
 * @return string          The pluralized duration string.
 */
if (!function_exists('pluralize')) {
	function pluralize($duration = 0)
	{
		if ($duration <= 1) {
			$pluralize = $duration . ' ' . 'day';
		} else {
			$pluralize = $duration . ' ' . 'days';
		}

		return $pluralize;
	}
}

/**
 * Get the start and end dates based on the selected date type.
 *
 * @param  string  $type  The selected date type.
 * @return array          An array containing the start and end dates formatted as 'Y-m-d H:i:s'.
 */
if (!function_exists('getDates')) {
	function getDates($type = "today")
	{
		$date1 = null;
		$date2 = null;

		$current_date = Carbon::now();

		// DETERMINE DATES BASED ON THE SELECTED DATE TYPE
		switch ($type) {
			case 'today':
				$date1 = $current_date->copy()->startOfDay();
				$date2 = $current_date->copy()->endOfDay();
				break;

			case 'this_week':
				$date1 = $current_date->copy()->startOfWeek();
				$date2 = $current_date->copy()->endOfWeek();
				break;

			case 'this_month':
				$date1 = $current_date->copy()->startOfMonth();
				$date2 = $current_date->copy()->endOfMonth();
				break;

			case 'last_month':
				$date1 = $current_date->copy()->subMonth()->startOfMonth();
				$date2 = $current_date->copy()->subMonth()->endOfMonth();
				break;

			case 'last_three_months':
				$date1 = $current_date->copy()->subMonths(2)->startOfMonth();
				$date2 = $current_date->copy()->endOfMonth();
				break;

			case 'last_six_months':
				$date1 = $current_date->copy()->subMonths(5)->startOfMonth();
				$date2 = $current_date->copy()->endOfMonth();
				break;

			case 'this_year':
				$date1 = $current_date->copy()->startOfYear();
				$date2 = $current_date->copy()->endOfYear();
				break;

			case 'last_year':
				$date1 = $current_date->copy()->subYear()->startOfYear();
				$date2 = $current_date->copy()->subYear()->endOfYear();
				break;

			default:
				$date1 = $current_date->copy()->startOfMonth();
				$date2 = $current_date->copy()->endOfMonth();
				break;
		}

		if ($date1 !== null) {
			$date1 = $date1->format('Y-m-d H:i:s');
		}

		if ($date2 !== null) {
			$date2 = $date2->format('Y-m-d H:i:s');
		}

		return [$date1, $date2];
	}
}

/**
 * Get the name associated with the transaction.
 *
 * @param  mixed  $transaction  The transaction object.
 * @return string               The name associated with the transaction.
 */
if (!function_exists('transaction_name')) {
	function transaction_name($transaction)
	{
		$name = "";
		if(!empty($transaction)) {
			if ($transaction->transactionable_type == Membership::class) {
				$member_id = $transaction->transactionable->member_id;
				if(!empty($member_id)) { 
					$name = Member::find($member_id)->name;
				}
			} else {
				$name = $transaction->transactionable->name;
			}
		}

		return $name;
	}
}
 
 